import React, { useState, useEffect } from "react";
import axios from 'axios';
import Form from "@rjsf/core";
import validator from "@rjsf/validator-ajv8";

// JSON schema for Pricing Rules Form
export const userSchema = {
  title: "Service Form",
  description: "Create or update Service.",
  type: "object",
  required: [
    "service_title",
    "service_description",
    "capacity_jobs",
    "has_cost",
    "is_vehicle_focused",
    "image",
    "map_icon",
  ],
  properties: {
    service_title: {
      type: "string",
      title: "Service Name",
    },
    service_description: {
      type: "string",
      title: "Service Description",
    },
    skills: {
      type: "array",
      title: "Skills",
      items: {
        type: "string",
        oneOf: [], // Will be populated dynamically
      },
      uniqueItems: true,
    },
    capacity_jobs: {
      type: "number",
      title: "Capacity",
    },
    has_cost: {
      type: "number", // Change to enum with 1 and 0 for radio
      title: "Has Cost",
      enum: [1, 0],
      enumNames: ["Yes", "No"],
    },
    is_vehicle_focused: {
      type: "number", // Change to enum with 1 and 0 for radio
      title: "Is Vehicle Focused",
      enum: [1, 0],
      enumNames: ["Yes", "No"],
    },    
    checklist: {
      type: "array",
      title: "Checklist",
      items: {
        type: "string",
        oneOf: [], // Will be populated dynamically
      },
      uniqueItems: true,
    },
    addons: {
      type: "array",
      title: "Addons",
      items: {
        type: "string",
        oneOf: [], // Will be populated dynamically
      },
      uniqueItems: true,
    },
    image: {
      type: "string",
      title: "Service Image",
      format: "data-url",
    },
    map_icon: {
      type: "string",
      title: "Map Icon",
      format: "data-url",
    },
    account_id: {
      type: "string", // Use string or number based on your DB
      title: "User ID",
      default: "{id}",
    },
  },
};

const CustomFileWidget = (props) => {
  const { value, onChange } = props;

  const handleChange = (e) => {
    const file = e.target.files[0];
    if (file) {
      const reader = new FileReader();
      reader.onload = (event) => {
        onChange(event.target.result);
      };
      reader.readAsDataURL(file);
    }
  };

  return (
    <div>
      {value && (
        <div>
          <img 
            src={value} 
            alt="Preview" 
            style={{ maxWidth: '100px', marginBottom: '10px' }} 
          />
          <p>Current: {value.startsWith('data:') ? 'New Upload' : value}</p>
        </div>
      )}
      <input 
        type="file" 
        onChange={handleChange} 
        accept="image/*" 
      />
    </div>
  );
};

// Optional UI schema tweaks
export const userUiSchema = {
  image: {
    "ui:widget": "file", // File upload for images
  },
  map_icon: {
    "ui:widget": "file", // File upload for map icons
  },
  is_vehicle_focused: {
    "ui:widget": "radio", // Set as radio button
  },
  has_cost: {
    "ui:widget": "radio", // Set as radio button
  },
  user_id: {
    "ui:widget": "hidden", // Hide user_id field
  },
  account_id: {
    "ui:widget": "hidden", // Hide user_id field
  },
  checklist: {
    "ui:widget": "select",
    "ui:options": {
      multiple: true,
      // Optional: Add placeholder text
      placeholder: "Select checklists...",
    },
  },
  skills: {
    "ui:widget": "select",
    "ui:options": {
      multiple: true,
      // Optional: Add placeholder text
      placeholder: "Select skills...",
    },
  },
  addons: {
    "ui:widget": "select",
    "ui:options": {
      multiple: true,
      // Optional: Add placeholder text
      placeholder: "Select addons...",
    },
  }
};

const UserForm = ({ initialData, onSubmit }) => {
  const [formData, setFormData] = useState(initialData || {});
  const [schema, setSchema] = useState(userSchema);

  
  useEffect(() => {
    const fetchChecklist = async () => {
      try {
        const userId = localStorage.getItem('userId');
        const response = await axios.get(`/services/getchecklist/${userId}`);

        if (response.data?.users) {
          const checkList = response.data.users.map((user) => ({
            const: String(user.user_id),
            title: user.checklist_name,
          }));

          setSchema(prev => ({
            ...prev,
            properties: {
              ...prev.properties,
              checklist: {
                ...prev.properties.checklist,
                items: {
                  ...prev.properties.checklist.items,
                  oneOf: checkList,
                },
              },
            },
          }));
        }
      } catch (error) {
        console.error("Error fetching checklists:", error);
      }
    };

    fetchChecklist();
  }, []);

  useEffect(() => {
    const fetchSkills = async () => {
      try {
        const userId = localStorage.getItem('userId');
        const response = await axios.get(`/services/getskills/${userId}`);

        if (response.data?.users) {
          const skills = response.data.users.map((user) => ({
            const: String(user.user_id),
            title: user.name,
          }));

          setSchema(prev => ({
            ...prev,
            properties: {
              ...prev.properties,
              skills: {
                ...prev.properties.skills,
                items: {
                  ...prev.properties.skills.items,
                  oneOf: skills,
                },
              },
            },
          }));
        }
      } catch (error) {
        console.error("Error fetching skills:", error);
      }
    };

    fetchSkills();
  }, []);

  useEffect(() => {
    const fetchAddons = async () => {
      try {
        const userId = localStorage.getItem('userId');
        const response = await axios.get(`/services/getaddonlist/${userId}`);

        if (response.data?.users) {
          const addons = response.data.users.map((user) => ({
            const: String(user.user_id),
            title: user.name, // Matches API's "name" field
          }));

          setSchema(prev => ({
            ...prev,
            properties: {
              ...prev.properties,
              addons: {
                ...prev.properties.addons,
                items: {
                  ...prev.properties.addons.items,
                  oneOf: addons,
                },
              },
            },
          }));
        }
      } catch (error) {
        console.error("Error fetching addons:", error);
      }
    };

    fetchAddons();
  }, []);
  


  const handleSubmit = ({ formData }) => {
    console.log("Form Submitted:", formData);
    onSubmit(formData);
  };

  return (
    <div style={{ maxWidth: "600px", margin: "0 auto" }}>
      <Form
        schema={schema}
        uiSchema={userUiSchema}
        formData={formData}
        onChange={({ formData }) => setFormData(formData)}
        onSubmit={handleSubmit}
        validator={validator}
        liveValidate
        widgets={{ FileWidget: CustomFileWidget }}
      />
    </div>
  );
};

export default UserForm;
